<?php
/**
 * Handles plugin activation, deactivation, and schema management.
 * 
 * Creates tables for:
 * - Event logging (basic detections)
 * - Intelligence collection (rich behavioral data)
 * - Agent fingerprints (for the global database)
 * - Rate limiting
 */

declare(strict_types=1);

if (!defined('ABSPATH')) {
    exit;
}

class Badger_Activator
{
    /** @var string Database schema version */
    private const DB_VERSION = '1.1.0';

    public static function activate(): void
    {
        self::create_tables();
        self::create_options();
        
        // Schedule intelligence sync
        if (!wp_next_scheduled('badger_intelligence_sync')) {
            wp_schedule_event(time(), 'five_minutes', 'badger_intelligence_sync');
        }
    }

    public static function deactivate(): void
    {
        // Clear scheduled events
        wp_clear_scheduled_hook('badger_intelligence_sync');
    }

    /**
     * Check and update database schema if needed.
     */
    public static function maybe_update_schema(): void
    {
        $current_version = get_option('badger_db_version', '0.0.0');
        
        if (version_compare($current_version, self::DB_VERSION, '<')) {
            self::create_tables();
            update_option('badger_db_version', self::DB_VERSION);
        }
    }

    private static function create_tables(): void
    {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        // Table 1: Basic events (for site owner dashboard)
        $events_table = $wpdb->prefix . 'badger_events';
        $sql_events = "CREATE TABLE IF NOT EXISTS $events_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            event_type varchar(50) NOT NULL,
            agent_id varchar(100) DEFAULT NULL,
            agent_name varchar(100) DEFAULT NULL,
            user_agent text DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            request_uri text DEFAULT NULL,
            referer text DEFAULT NULL,
            metadata longtext DEFAULT NULL,
            PRIMARY KEY (id),
            KEY created_at (created_at),
            KEY event_type (event_type),
            KEY agent_id (agent_id)
        ) $charset_collate;";
        dbDelta($sql_events);

        // Table 2: Intelligence (rich behavioral data for the network)
        $intel_table = $wpdb->prefix . 'badger_intelligence';
        $sql_intel = "CREATE TABLE IF NOT EXISTS $intel_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            agent_id varchar(100) NOT NULL,
            agent_name varchar(100) NOT NULL,
            user_agent varchar(500) DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            ip_version tinyint(1) DEFAULT 4,
            country varchar(2) DEFAULT NULL,
            asn varchar(50) DEFAULT NULL,
            is_datacenter tinyint(1) DEFAULT 0,
            is_vpn tinyint(1) DEFAULT 0,
            request_uri text DEFAULT NULL,
            request_method varchar(10) DEFAULT 'GET',
            referer varchar(500) DEFAULT NULL,
            event_type varchar(50) DEFAULT 'detection',
            behavioral_score decimal(3,2) DEFAULT 0.00,
            fingerprint_confidence decimal(3,2) DEFAULT 0.00,
            honeypot_depth tinyint(3) DEFAULT 0,
            trap_id varchar(100) DEFAULT NULL,
            session_id varchar(64) DEFAULT NULL,
            request_sequence int(11) DEFAULT 1,
            day_of_week tinyint(1) DEFAULT NULL,
            hour_of_day tinyint(2) DEFAULT NULL,
            site_type varchar(50) DEFAULT 'content',
            metadata longtext DEFAULT NULL,
            synced tinyint(1) DEFAULT 0,
            synced_at datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_agent_id (agent_id),
            KEY idx_ip_address (ip_address),
            KEY idx_created_at (created_at),
            KEY idx_behavioral_score (behavioral_score),
            KEY idx_honeypot_depth (honeypot_depth),
            KEY idx_event_type (event_type),
            KEY idx_synced (synced),
            KEY idx_session (session_id, request_sequence),
            KEY idx_country (country),
            KEY idx_composite (agent_id, ip_address, created_at)
        ) $charset_collate;";
        dbDelta($sql_intel);

        // Table 3: Agent fingerprints (discovered patterns)
        $fingerprints_table = $wpdb->prefix . 'badger_fingerprints';
        $sql_fingerprints = "CREATE TABLE IF NOT EXISTS $fingerprints_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            fingerprint_hash varchar(64) NOT NULL,
            agent_id varchar(100) DEFAULT 'unknown',
            agent_name varchar(100) DEFAULT 'Unknown Agent',
            user_agent_pattern varchar(500) DEFAULT NULL,
            header_fingerprint varchar(255) DEFAULT NULL,
            behavior_signature varchar(255) DEFAULT NULL,
            confidence_score decimal(3,2) DEFAULT 0.00,
            occurrence_count int(11) DEFAULT 1,
            first_seen datetime DEFAULT CURRENT_TIMESTAMP,
            last_seen datetime DEFAULT CURRENT_TIMESTAMP,
            sample_ips longtext DEFAULT NULL,
            sample_countries longtext DEFAULT NULL,
            is_verified tinyint(1) DEFAULT 0,
            metadata longtext DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY idx_fingerprint_hash (fingerprint_hash),
            KEY idx_agent_id (agent_id),
            KEY idx_confidence (confidence_score),
            KEY idx_last_seen (last_seen),
            KEY idx_verified (is_verified)
        ) $charset_collate;";
        dbDelta($sql_fingerprints);

        // Table 4: Rate limiting
        $rate_table = $wpdb->prefix . 'badger_rate_limits';
        $sql_rate = "CREATE TABLE IF NOT EXISTS $rate_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            identifier varchar(255) NOT NULL,
            identifier_type varchar(20) NOT NULL DEFAULT 'ip',
            request_count int(11) NOT NULL DEFAULT 1,
            window_start datetime NOT NULL,
            first_request_uri text DEFAULT NULL,
            last_request_uri text DEFAULT NULL,
            blocked tinyint(1) DEFAULT 0,
            blocked_until datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY idx_identifier_window (identifier, window_start),
            KEY idx_blocked (blocked),
            KEY idx_blocked_until (blocked_until)
        ) $charset_collate;";
        dbDelta($sql_rate);

        // Table 5: Network contribution stats (for site owner pride)
        $stats_table = $wpdb->prefix . 'badger_network_stats';
        $sql_stats = "CREATE TABLE IF NOT EXISTS $stats_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            stat_date date NOT NULL,
            detections_local int(11) DEFAULT 0,
            detections_contributed int(11) DEFAULT 0,
            unique_agents int(11) DEFAULT 0,
            honeypot_triggers int(11) DEFAULT 0,
            new_fingerprints int(11) DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY idx_date (stat_date)
        ) $charset_collate;";
        dbDelta($sql_stats);

        update_option('badger_db_version', self::DB_VERSION);
    }

    private static function create_options(): void
    {
        $defaults = [
            // Core settings
            'badger_honeypot_enabled' => true,
            'badger_blocked_agents' => [],
            'badger_block_all_unknown' => false,
            'badger_robots_txt_disallow' => true,
            
            // Login protection
            'badger_login_protection' => true,
            'badger_login_max_attempts' => 5,
            'badger_login_lockout_minutes' => 15,
            
            // Security headers
            'badger_security_headers' => true,
            'badger_csp_header' => '',
            'badger_disable_file_editor' => true,
            'badger_disable_xmlrpc' => true,
            'badger_protect_user_enumeration' => true,
            
            // Rate limiting
            'badger_rate_limit_enabled' => false,
            'badger_rate_limit_requests' => 60,
            'badger_rate_limit_window' => 60,
            
            // Intelligence collection
            'badger_intelligence_enabled' => true,
            'badger_intelligence_sync_enabled' => true,
            'badger_intelligence_anonymize_ips' => true, // Hash IPs before transmission
            'badger_site_category' => 'general',
            
            // Network participation
            'badger_site_uuid' => wp_generate_uuid4(),
            'badger_intelligence_stats' => [
                'total_contributed' => 0,
                'last_contribution' => 0,
                'unique_agents_seen' => [],
            ],
        ];

        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }
    }

    /**
     * Get database schema info for debugging.
     */
    public static function get_schema_info(): array
    {
        global $wpdb;
        
        $tables = [
            'badger_events',
            'badger_intelligence',
            'badger_fingerprints',
            'badger_rate_limits',
            'badger_network_stats',
        ];
        
        $info = [
            'db_version' => get_option('badger_db_version', 'unknown'),
            'expected_version' => self::DB_VERSION,
            'tables' => [],
        ];
        
        foreach ($tables as $table) {
            $full_name = $wpdb->prefix . $table;
            $exists = $wpdb->get_var($wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $full_name
            )) === $full_name;
            
            $count = 0;
            if ($exists) {
                $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM $full_name");
            }
            
            $info['tables'][$table] = [
                'exists' => $exists,
                'rows' => $count,
            ];
        }
        
        return $info;
    }
}
