<?php
/**
 * Admin menu and settings.
 */

declare(strict_types=1);

if (!defined('ABSPATH')) {
    exit;
}

class Badger_Admin
{
    private static ?self $instance = null;

    public static function get_instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action('admin_menu', [$this, 'add_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    public function add_menu(): void
    {
        add_menu_page(
            'Badger',
            'Badger',
            'manage_options',
            'badger',
            [Badger_Dashboard::class, 'render'],
            'dashicons-shield',
            80
        );

        add_submenu_page(
            'badger',
            'Settings',
            'Settings',
            'manage_options',
            'badger-settings',
            [$this, 'render_settings']
        );
    }

    public function register_settings(): void
    {
        register_setting('badger', 'badger_honeypot_enabled', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);

        register_setting('badger', 'badger_blocked_agents', [
            'type' => 'array',
            'default' => [],
            'sanitize_callback' => function ($val) {
                return is_array($val) ? array_map('sanitize_text_field', $val) : [];
            },
        ]);

        register_setting('badger', 'badger_block_all_unknown', [
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);

        register_setting('badger', 'badger_robots_txt_disallow', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);

        register_setting('badger', 'badger_login_protection', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);
        register_setting('badger', 'badger_login_max_attempts', [
            'type' => 'integer',
            'default' => 5,
            'sanitize_callback' => function ($val) {
                return max(3, min(20, (int) $val));
            },
        ]);
        register_setting('badger', 'badger_login_lockout_minutes', [
            'type' => 'integer',
            'default' => 15,
            'sanitize_callback' => function ($val) {
                return max(5, min(1440, (int) $val));
            },
        ]);
        register_setting('badger', 'badger_security_headers', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);
        register_setting('badger', 'badger_csp_header', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_text_field',
        ]);
        register_setting('badger', 'badger_disable_file_editor', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);
        register_setting('badger', 'badger_disable_xmlrpc', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);
        register_setting('badger', 'badger_protect_user_enumeration', [
            'type' => 'boolean',
            'default' => true,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);
        register_setting('badger', 'badger_rate_limit_enabled', [
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ]);
        register_setting('badger', 'badger_rate_limit_requests', [
            'type' => 'integer',
            'default' => 60,
            'sanitize_callback' => function ($val) {
                return max(10, min(1000, (int) $val));
            },
        ]);
        register_setting('badger', 'badger_rate_limit_window', [
            'type' => 'integer',
            'default' => 60,
            'sanitize_callback' => function ($val) {
                return max(60, min(3600, (int) $val));
            },
        ]);
    }

    public function enqueue_assets(string $hook): void
    {
        if (strpos($hook, 'badger') === false) {
            return;
        }

        wp_enqueue_style(
            'badger-admin',
            BADGER_PLUGIN_URL . 'admin/css/admin.css',
            [],
            BADGER_VERSION
        );

        wp_enqueue_script(
            'badger-admin',
            BADGER_PLUGIN_URL . 'admin/js/admin.js',
            [],
            BADGER_VERSION,
            true
        );
    }

    public function render_settings(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'ai';
        $tabs = [
            'ai' => 'AI & Honeypot',
            'security' => 'Security',
        ];
        if (!isset($tabs[$tab])) {
            $tab = 'ai';
        }

        $honeypot_enabled = get_option('badger_honeypot_enabled', true);
        $blocked_agents = get_option('badger_blocked_agents', []);
        $block_all_unknown = get_option('badger_block_all_unknown', false);
        $robots_disallow = get_option('badger_robots_txt_disallow', true);

        $login_protection = get_option('badger_login_protection', true);
        $login_max_attempts = get_option('badger_login_max_attempts', 5);
        $login_lockout_min = get_option('badger_login_lockout_minutes', 15);
        $security_headers = get_option('badger_security_headers', true);
        $csp_header = get_option('badger_csp_header', '');
        $disable_file_editor = get_option('badger_disable_file_editor', true);
        $disable_xmlrpc = get_option('badger_disable_xmlrpc', true);
        $protect_enumeration = get_option('badger_protect_user_enumeration', true);
        $rate_limit_enabled = get_option('badger_rate_limit_enabled', false);
        $rate_limit_requests = get_option('badger_rate_limit_requests', 60);
        $rate_limit_window = get_option('badger_rate_limit_window', 60);

        $detector = Badger_Detector::get_instance();
        $crawlers = $detector->get_crawlers();
        ?>
        <div class="wrap badger-settings">
            <h1>Badger Settings</h1>

            <nav class="nav-tab-wrapper">
                <?php foreach ($tabs as $key => $label): ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=badger-settings&tab=' . $key)); ?>"
                        class="nav-tab <?php echo $tab === $key ? 'nav-tab-active' : ''; ?>"><?php echo esc_html($label); ?></a>
                <?php endforeach; ?>
            </nav>

            <form method="post" action="options.php">
                <?php settings_fields('badger'); ?>

                <?php if ($tab === 'ai'): ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Honeypot Traps</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_honeypot_enabled" value="1"
                                    <?php checked($honeypot_enabled); ?> />
                                Enable honeypot trap pages
                            </label>
                            <p class="description">Invisible links that AI agents follow. When they do, we log them.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Block AI Agents</th>
                        <td>
                            <?php foreach ($crawlers as $crawler): ?>
                                <label class="block">
                                    <input type="checkbox" name="badger_blocked_agents[]" value="<?php echo esc_attr($crawler['id']); ?>"
                                        <?php checked(in_array($crawler['id'], $blocked_agents, true)); ?> />
                                    <?php echo esc_html($crawler['name']); ?>
                                </label>
                            <?php endforeach; ?>
                            <p class="description">Blocked agents receive 403 Forbidden.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Block Unknown</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_block_all_unknown" value="1"
                                    <?php checked($block_all_unknown); ?> />
                                Block all unidentified AI crawlers
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">robots.txt</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_robots_txt_disallow" value="1"
                                    <?php checked($robots_disallow); ?> />
                                Add Disallow rules for AI crawlers to robots.txt
                            </label>
                        </td>
                    </tr>
                </table>
                <?php endif; ?>

                <?php if ($tab === 'security'): ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Login Protection</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_login_protection" value="1"
                                    <?php checked($login_protection); ?> />
                                Limit login attempts and lockout after failures
                            </label>
                            <p class="description">Max attempts: <input type="number" name="badger_login_max_attempts"
                                value="<?php echo esc_attr((string) $login_max_attempts); ?>" min="3" max="20" /> &nbsp;
                                Lockout: <input type="number" name="badger_login_lockout_minutes"
                                value="<?php echo esc_attr((string) $login_lockout_min); ?>" min="5" max="1440" /> minutes</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Security Headers</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_security_headers" value="1"
                                    <?php checked($security_headers); ?> />
                                Send X-Frame-Options, X-Content-Type-Options, Referrer-Policy, Permissions-Policy
                            </label>
                            <p class="description">Optional CSP: <input type="text" name="badger_csp_header"
                                value="<?php echo esc_attr($csp_header); ?>" class="regular-text" placeholder="default-src 'self'" /></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">File Editor</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_disable_file_editor" value="1"
                                    <?php checked($disable_file_editor); ?> />
                                Disable theme and plugin file editor
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">XML-RPC</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_disable_xmlrpc" value="1"
                                    <?php checked($disable_xmlrpc); ?> />
                                Disable XML-RPC (common attack vector)
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">User Enumeration</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_protect_user_enumeration" value="1"
                                    <?php checked($protect_enumeration); ?> />
                                Block ?author=N user enumeration
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Rate Limiting</th>
                        <td>
                            <label>
                                <input type="checkbox" name="badger_rate_limit_enabled" value="1"
                                    <?php checked($rate_limit_enabled); ?> />
                                Throttle requests per IP
                            </label>
                            <p class="description">Max <input type="number" name="badger_rate_limit_requests"
                                value="<?php echo esc_attr((string) $rate_limit_requests); ?>" min="10" max="1000" />
                                requests per <input type="number" name="badger_rate_limit_window"
                                value="<?php echo esc_attr((string) $rate_limit_window); ?>" min="60" max="3600" /> seconds</p>
                        </td>
                    </tr>
                </table>
                <?php endif; ?>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}
