<?php
/**
 * Blocks requests from identified AI agents based on site policy.
 */

declare(strict_types=1);

if (!defined('ABSPATH')) {
    exit;
}

class Badger_Blocker
{
    private static ?self $instance = null;

    public static function get_instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action('init', [$this, 'maybe_block'], 0);
    }

    /**
     * Block request if agent is in blocklist.
     */
    public function maybe_block(): void
    {
        if (is_admin() || wp_doing_cron()) {
            return;
        }

        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        if (empty($user_agent)) {
            return;
        }

        $detector = Badger_Detector::get_instance();
        $agent = $detector->identify_agent($user_agent);

        if ($agent === null) {
            if (get_option('badger_block_all_unknown', false)) {
                $this->block('unknown');
            }
            return;
        }

        $blocked = get_option('badger_blocked_agents', []);
        if (is_array($blocked) && in_array($agent['id'], $blocked, true)) {
            $this->block($agent['id']);
        }
    }

    /**
     * Send 403 and exit.
     */
    private function block(string $agent_id): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'badger_events';

        $wpdb->insert(
            $table,
            [
                'event_type' => 'blocked',
                'agent_id' => $agent_id,
                'user_agent' => substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 500),
                'ip_address' => $this->get_client_ip(),
                'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
            ],
            ['%s', '%s', '%s', '%s', '%s']
        );

        status_header(403);
        wp_die('Forbidden', 'Forbidden', ['response' => 403]);
    }

    private function get_client_ip(): string
    {
        $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (str_contains($ip, ',')) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                return sanitize_text_field($ip);
            }
        }
        return '';
    }
}
