<?php
/**
 * Main dashboard view. Shows agent intelligence, events, and stats.
 */

declare(strict_types=1);

if (!defined('ABSPATH')) {
    exit;
}

class Badger_Dashboard
{
    public static function render(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'badger_events';

        $total_events = (int) $wpdb->get_var("SELECT COUNT(*) FROM $table");
        $detected = (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE event_type = 'detected'");
        $honeypot_hits = (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE event_type = 'honeypot_hit'");
        $blocked = (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE event_type = 'blocked'");

        $recent = $wpdb->get_results(
            "SELECT * FROM $table ORDER BY created_at DESC LIMIT 50",
            ARRAY_A
        );

        $by_agent = $wpdb->get_results(
            "SELECT agent_name, COUNT(*) as cnt FROM $table WHERE agent_name IS NOT NULL AND agent_name != '' GROUP BY agent_name ORDER BY cnt DESC",
            ARRAY_A
        );

        $security = Badger_Hardening::get_security_status();
        $security_enabled = array_sum(array_map('intval', $security));
        $security_total = count($security);
        ?>
        <div class="wrap badger-dashboard">
            <h1>Badger — Agent Intelligence</h1>
            <p class="badger-tagline">Don't just block AI bots. Trap them.</p>

            <div class="badger-stats">
                <div class="badger-stat">
                    <span class="badger-stat-value"><?php echo esc_html(number_format($detected)); ?></span>
                    <span class="badger-stat-label">Detected</span>
                </div>
                <div class="badger-stat">
                    <span class="badger-stat-value"><?php echo esc_html(number_format($honeypot_hits)); ?></span>
                    <span class="badger-stat-label">Honeypot Hits</span>
                </div>
                <div class="badger-stat">
                    <span class="badger-stat-value"><?php echo esc_html(number_format($blocked)); ?></span>
                    <span class="badger-stat-label">Blocked</span>
                </div>
            </div>

            <div class="badger-section badger-security-status">
                <h2>Security Hardening</h2>
                <p class="badger-muted"><?php echo esc_html((string) $security_enabled); ?> of <?php echo esc_html((string) $security_total); ?> protections active — <a href="<?php echo esc_url(admin_url('admin.php?page=badger-settings&tab=security')); ?>">Configure</a></p>
                <ul class="badger-checklist">
                    <li class="<?php echo $security['login_protection'] ? 'on' : 'off'; ?>">Login protection</li>
                    <li class="<?php echo $security['security_headers'] ? 'on' : 'off'; ?>">Security headers</li>
                    <li class="<?php echo $security['file_editor_disabled'] ? 'on' : 'off'; ?>">File editor disabled</li>
                    <li class="<?php echo $security['xmlrpc_disabled'] ? 'on' : 'off'; ?>">XML-RPC disabled</li>
                    <li class="<?php echo $security['user_enumeration_protected'] ? 'on' : 'off'; ?>">User enumeration blocked</li>
                    <li class="<?php echo $security['rate_limit_enabled'] ? 'on' : 'off'; ?>">Rate limiting</li>
                </ul>
            </div>

            <?php if (!empty($by_agent)): ?>
            <div class="badger-section">
                <h2>Agents by Volume</h2>
                <table class="badger-table">
                    <thead>
                        <tr>
                            <th>Agent</th>
                            <th>Events</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($by_agent as $row): ?>
                        <tr>
                            <td><?php echo esc_html($row['agent_name']); ?></td>
                            <td><?php echo esc_html(number_format((int) $row['cnt'])); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>

            <div class="badger-section">
                <h2>Recent Events</h2>
                <table class="badger-table">
                    <thead>
                        <tr>
                            <th>Time</th>
                            <th>Type</th>
                            <th>Agent</th>
                            <th>IP</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent as $event): ?>
                        <tr>
                            <td><?php echo esc_html($event['created_at']); ?></td>
                            <td><span class="badger-badge badger-badge-<?php echo esc_attr($event['event_type']); ?>"><?php echo esc_html($event['event_type']); ?></span></td>
                            <td><?php echo esc_html($event['agent_name'] ?? '-'); ?></td>
                            <td><code><?php echo esc_html($event['ip_address'] ?? '-'); ?></code></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php if (empty($recent)): ?>
                <p class="badger-muted">No events yet. Badger is watching.</p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
}
