<?php
/**
 * Security hardening: login protection, headers, XML-RPC, file editor, user enumeration.
 */

declare(strict_types=1);

if (!defined('ABSPATH')) {
    exit;
}

class Badger_Hardening
{
    private static ?self $instance = null;

    public static function get_instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action('init', [$this, 'maybe_block_user_enumeration'], 1);
        add_action('send_headers', [$this, 'maybe_send_security_headers']);
        add_action('wp_login_failed', [$this, 'handle_login_failed'], 10, 2);
        add_action('wp_login', [$this, 'clear_login_failures'], 10, 2);
        add_action('wp_authenticate', [$this, 'maybe_block_login'], 20, 2);
        add_action('admin_menu', [self::class, 'remove_editor_menus'], 999);
        add_action('load-theme-editor.php', [$this, 'maybe_block_theme_editor']);
        add_action('load-plugin-editor.php', [$this, 'maybe_block_plugin_editor']);
        add_filter('xmlrpc_enabled', [$this, 'maybe_disable_xmlrpc_filter']);
    }

    /**
     * Block user enumeration via ?author=N.
     */
    public function maybe_block_user_enumeration(): void
    {
        if (!get_option('badger_protect_user_enumeration', true)) {
            return;
        }

        $author = isset($_GET['author']) ? (int) $_GET['author'] : 0;
        if ($author > 0) {
            $this->deny_request('User enumeration blocked');
        }
    }

    /**
     * Send security headers when configured.
     */
    public function maybe_send_security_headers(): void
    {
        if (!get_option('badger_security_headers', true)) {
            return;
        }

        if (is_admin() || wp_doing_cron()) {
            return;
        }

        $headers = [
            'X-Frame-Options' => 'SAMEORIGIN',
            'X-Content-Type-Options' => 'nosniff',
            'Referrer-Policy' => 'strict-origin-when-cross-origin',
            'Permissions-Policy' => 'geolocation=(), microphone=(), camera=()',
        ];

        $csp = get_option('badger_csp_header', '');
        if ($csp !== '') {
            $headers['Content-Security-Policy'] = $csp;
        }

        foreach ($headers as $name => $value) {
            header(sprintf('%s: %s', $name, $value));
        }
    }

    /**
     * Track failed login attempts and trigger lockout.
     */
    public function handle_login_failed(string $username, WP_Error $error): void
    {
        if (!get_option('badger_login_protection', true)) {
            return;
        }

        $ip = $this->get_client_ip();
        $key = 'badger_login_fail_' . md5($ip);
        $attempts = (int) get_transient($key);
        $attempts++;

        $max = (int) get_option('badger_login_max_attempts', 5);
        $lockout_min = (int) get_option('badger_login_lockout_minutes', 15);

        if ($attempts >= $max) {
            set_transient('badger_login_blocked_' . md5($ip), true, $lockout_min * MINUTE_IN_SECONDS);
            delete_transient($key);
        } else {
            set_transient($key, $attempts, 15 * MINUTE_IN_SECONDS);
        }
    }

    /**
     * Clear failed login count on successful login.
     */
    public function clear_login_failures(string $username, WP_User $user): void
    {
        if (!get_option('badger_login_protection', true)) {
            return;
        }

        $ip = $this->get_client_ip();
        delete_transient('badger_login_fail_' . md5($ip));
    }

    /**
     * Block login if IP is in lockout.
     */
    public function maybe_block_login($username, $password): void
    {
        if (!get_option('badger_login_protection', true)) {
            return;
        }

        $ip = $this->get_client_ip();
        $key = 'badger_login_blocked_' . md5($ip);

        if (get_transient($key)) {
            wp_die(
                esc_html__('Too many failed login attempts. Please try again later.', 'badger'),
                esc_html__('Login Blocked', 'badger'),
                ['response' => 403]
            );
        }
    }

    /**
     * Block theme editor access when disabled.
     */
    public function maybe_block_theme_editor(): void
    {
        if (get_option('badger_disable_file_editor', true)) {
            wp_die(
                esc_html__('File editing is disabled for security.', 'badger'),
                '',
                ['response' => 403]
            );
        }
    }

    /**
     * Block plugin editor access when disabled.
     */
    public function maybe_block_plugin_editor(): void
    {
        if (get_option('badger_disable_file_editor', true)) {
            wp_die(
                esc_html__('File editing is disabled for security.', 'badger'),
                '',
                ['response' => 403]
            );
        }
    }

    /**
     * Disable XML-RPC when configured.
     */
    public function maybe_disable_xmlrpc_filter(bool $enabled): bool
    {
        if (get_option('badger_disable_xmlrpc', true)) {
            return false;
        }
        return $enabled;
    }

    /**
     * Block request with 403.
     */
    private function deny_request(string $message): void
    {
        status_header(403);
        wp_die(esc_html($message), 'Forbidden', ['response' => 403]);
    }

    private function get_client_ip(): string
    {
        $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (str_contains($ip, ',')) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                return sanitize_text_field($ip);
            }
        }
        return '';
    }

    /**
     * Remove file editor menu items when disabled.
     */
    public static function remove_editor_menus(): void
    {
        if (!get_option('badger_disable_file_editor', true)) {
            return;
        }

        remove_submenu_page('themes.php', 'theme-editor.php');
        remove_submenu_page('plugins.php', 'plugin-editor.php');
    }

    /**
     * Return security hardening status for dashboard display.
     */
    public static function get_security_status(): array
    {
        return [
            'login_protection' => get_option('badger_login_protection', true),
            'security_headers' => get_option('badger_security_headers', true),
            'file_editor_disabled' => get_option('badger_disable_file_editor', true),
            'xmlrpc_disabled' => get_option('badger_disable_xmlrpc', true),
            'user_enumeration_protected' => get_option('badger_protect_user_enumeration', true),
            'rate_limit_enabled' => get_option('badger_rate_limit_enabled', false),
        ];
    }
}
