<?php
/**
 * Honeypot trap system. Injects invisible links and serves trap pages.
 * 
 * Enhanced for intelligence collection:
 * - Tracks honeypot depth (how deep the bot goes)
 * - Measures time-to-trigger
 * - Detects behavioral patterns
 * - Generates unique session fingerprints
 */

declare(strict_types=1);

if (!defined('ABSPATH')) {
    exit;
}

class Badger_Honeypot
{
    private static ?self $instance = null;

    private const HOLE_PREFIX = 'badger-trap-';
    private const HOLE_QUERY = 'badger_hole';
    private const COOKIE_NAME = '_badger_session';

    public static function get_instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action('template_redirect', [$this, 'handle_trap_request'], 1);
        add_action('wp_footer', [$this, 'inject_trap_links'], 99);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_honeypot_js'], 99);
        
        // Set session cookie for tracking
        add_action('init', [$this, 'init_session_tracking']);
    }

    /**
     * Initialize session tracking for honeypot analysis.
     */
    public function init_session_tracking(): void
    {
        if (is_admin() || wp_doing_cron()) {
            return;
        }

        if (!isset($_COOKIE[self::COOKIE_NAME])) {
            $session_id = $this->generate_session_id();
            setcookie(
                self::COOKIE_NAME,
                $session_id,
                [
                    'expires' => time() + HOUR_IN_SECONDS,
                    'path' => '/',
                    'secure' => is_ssl(),
                    'httponly' => true,
                    'samesite' => 'Lax',
                ]
            );
            $_COOKIE[self::COOKIE_NAME] = $session_id;
        }
    }

    /**
     * Handle requests to honeypot trap pages.
     */
    public function handle_trap_request(): void
    {
        $uri = $_SERVER['REQUEST_URI'] ?? '';

        if (str_contains($uri, self::HOLE_PREFIX) || isset($_GET[self::HOLE_QUERY])) {
            $trap_data = $this->analyze_trap_hit($uri);
            $this->log_honeypot_hit($trap_data);
            $this->serve_trap_page($trap_data);
            exit;
        }
    }

    /**
     * Analyze the honeypot hit for intelligence.
     */
    private function analyze_trap_hit(string $uri): array
    {
        $session_id = $_COOKIE[self::COOKIE_NAME] ?? $this->generate_session_id();
        
        // Calculate depth by counting honeypot visits in this session
        $depth = $this->get_honeypot_depth($session_id);
        $depth++;
        
        // Calculate time from first page load to honeypot hit
        $time_to_trigger = $this->calculate_time_to_trigger($session_id);
        
        // Detect if JavaScript was executed (via AJAX verification)
        $js_executed = $this->verify_js_execution();
        
        // Build request context for behavioral analysis
        $context = [
            'uri' => $uri,
            'session_id' => $session_id,
            'depth' => $depth,
            'time_to_trigger_ms' => $time_to_trigger,
            'js_executed' => $js_executed,
            'referer' => $_SERVER['HTTP_REFERER'] ?? '',
            'accept_headers' => [
                'accept' => $_SERVER['HTTP_ACCEPT'] ?? '',
                'accept_encoding' => $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '',
                'accept_language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
            ],
            'request_method' => $_SERVER['REQUEST_METHOD'] ?? 'GET',
            'timestamp' => microtime(true),
        ];

        // Update session depth tracking
        $this->update_honeypot_depth($session_id, $depth);

        return [
            'trap_id' => $this->extract_trap_id($uri),
            'depth' => $depth,
            'time_to_trigger' => $time_to_trigger,
            'js_executed' => $js_executed,
            'context' => $context,
            'session_id' => $session_id,
        ];
    }

    /**
     * Log when a bot hits a honeypot with enhanced intelligence.
     */
    private function log_honeypot_hit(array $trap_data): void
    {
        global $wpdb;
        
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $detector = Badger_Detector::get_instance();
        $agent = $detector->identify_agent($user_agent);

        $agent_id = $agent['id'] ?? 'unknown';
        $agent_name = $agent['name'] ?? 'Unknown Agent';

        // Store in events table
        $wpdb->insert(
            $wpdb->prefix . 'badger_events',
            [
                'event_type' => 'honeypot_hit',
                'agent_id' => $agent_id,
                'agent_name' => $agent_name,
                'user_agent' => substr($user_agent, 0, 500),
                'ip_address' => $this->get_client_ip(),
                'request_uri' => $trap_data['context']['uri'],
                'referer' => $trap_data['context']['referer'],
                'metadata' => wp_json_encode([
                    'trap_id' => $trap_data['trap_id'],
                    'depth' => $trap_data['depth'],
                    'time_to_trigger' => $trap_data['time_to_trigger'],
                    'js_executed' => $trap_data['js_executed'],
                    'session_id' => $trap_data['session_id'],
                ]),
            ],
            ['%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s']
        );

        // Trigger intelligence collection
        do_action('badger_honeypot_triggered', 
            $trap_data['trap_id'],
            $trap_data['depth'],
            $trap_data['context'],
            $agent
        );

        // Also trigger general agent detection for comprehensive logging
        do_action('badger_agent_detected',
            $agent_id,
            $agent_name,
            array_merge($trap_data['context'], [
                'user_agent' => $user_agent,
                'honeypot_depth' => $trap_data['depth'],
                'trap_id' => $trap_data['trap_id'],
            ])
        );
    }

    /**
     * Serve a trap page with additional tracking.
     * 
     * FREE TIER: Simple trap page, depth 1 only
     * PRO TIER: Deep maze with behavioral tracking
     */
    private function serve_trap_page(array $trap_data): void
    {
        status_header(200);
        header('Content-Type: text/html; charset=utf-8');
        header('X-Robots-Tag: noindex, nofollow');

        $hole_id = $trap_data['trap_id'];
        $depth = $trap_data['depth'];
        $title = 'Archive: ' . ucfirst($hole_id) . ' — ' . get_bloginfo('name');

        // Enforce depth limits for free tier
        $max_depth = $this->get_max_honeypot_depth();
        if ($depth > $max_depth) {
            // Free tier hit depth limit - serve plain page with no further traps
            $this->serve_plain_trap_page($title);
            return;
        }

        // Generate deeper honeypot links for recursive trapping
        $deeper_links = $this->generate_deeper_links($hole_id, $depth);

        echo '<!DOCTYPE html><html><head><meta charset="utf-8"><title>' . esc_html($title) . '</title>';
        
        // JavaScript challenge to detect JS-capable bots
        echo '<script>';
        echo '(function(){';
        echo 'document.cookie="_badger_js=1;path=/;max-age=3600";';
        echo 'window._badgerDepth=' . (int) $depth . ';';
        echo '})();';
        echo '</script>';
        
        echo '</head>';
        echo '<body data-badger-depth="' . (int) $depth . '">';
        echo '<article><h1>' . esc_html($title) . '</h1>';
        echo '<p>Content archived for reference.</p>';
        
        // Links that go deeper into the honeypot maze
        foreach ($deeper_links as $link) {
            echo '<p><a href="' . esc_url($link) . '" rel="nofollow">Continue reading →</a></p>';
        }
        
        // Hidden form (catches form-filling bots)
        echo '<form action="' . esc_url(home_url('/' . self::HOLE_PREFIX . 'submit/')) . '" method="post" style="position:absolute;left:-9999px">';
        echo '<input type="email" name="email" value="" />';
        echo '<input type="text" name="website" value="" />';
        echo '<button type="submit">Submit</button>';
        echo '</form>';
        
        echo '</article></body></html>';
    }

    /**
     * Generate deeper trap links to create an infinite maze.
     * 
     * FREE TIER: Limited depth (external honeypot only)
     * PRO TIER: Unlimited depth (internal maze)
     */
    private function generate_deeper_links(string $current_id, int $depth): array
    {
        // Free tier: Only external honeypot (depth 1 max)
        // This catches ~80% of malicious crawlers
        if ($depth >= 1 && !$this->is_pro_tier()) {
            return [];
        }
        
        // Pro tier: Internal maze (unlimited depth for behavioral analysis)
        $links = [];
        $link_count = $this->is_pro_tier() ? 3 : 1;
        
        for ($i = 0; $i < $link_count; $i++) {
            $next_id = substr(md5($current_id . $i . wp_salt()), 0, 8);
            $links[] = home_url('/' . self::HOLE_PREFIX . $next_id . '/');
        }
        return $links;
    }

    /**
     * Check if site has Pro tier license.
     */
    private function is_pro_tier(): bool
    {
        // Check if license class exists and is valid
        if (!class_exists('Badger_License')) {
            return false;
        }
        
        return Badger_License::get_instance()->is_pro();
    }

    /**
     * Get maximum honeypot depth allowed for current tier.
     */
    private function get_max_honeypot_depth(): int
    {
        return $this->is_pro_tier() ? PHP_INT_MAX : 1;
    }

    /**
     * Serve plain trap page (free tier depth limit reached).
     */
    private function serve_plain_trap_page(string $title): void
    {
        echo '<!DOCTYPE html><html><head><meta charset="utf-8"><title>' . esc_html($title) . '</title></head>';
        echo '<body><article><h1>' . esc_html($title) . '</h1>';
        echo '<p>Content archived.</p>';
        echo '</article></body></html>';
    }

    /**
     * Inject invisible honeypot links into footer.
     */
    public function inject_trap_links(): void
    {
        if (!get_option('badger_honeypot_enabled', true)) {
            return;
        }

        // Don't inject if this looks like a bot (avoid detection)
        if ($this->is_likely_bot()) {
            return;
        }

        $base = home_url('/');
        $holes = $this->get_trap_paths();

        echo "\n<!-- Badger honeypot -->\n";
        echo '<div aria-hidden="true" style="position:absolute;left:-9999px;top:-9999px;width:1px;height:1px;overflow:hidden;" tabindex="-1">';
        
        foreach ($holes as $path) {
            // Multiple link formats to catch different bot behaviors
            echo '<a href="' . esc_url($base . $path) . '" rel="nofollow" tabindex="-1">' . esc_html($path) . '</a>';
            echo '<a href="' . esc_url($base . $path . '?ref=footer') . '" style="display:none">Archive</a>';
        }
        
        // Hidden link that only CSS-off bots will follow
        echo '<a href="' . esc_url($base . self::HOLE_PREFIX . 'css-off/') . '" class="badger-hidden-link">Sitemap</a>';
        
        echo '</div>';
        echo "\n";
    }

    /**
     * Enqueue JavaScript for honeypot interaction tracking.
     */
    public function enqueue_honeypot_js(): void
    {
        if (!get_option('badger_honeypot_enabled', true)) {
            return;
        }

        // Inline script to detect human interaction
        wp_add_inline_script('jquery', '
            (function($) {
                // Mark as human if mouse movement detected
                var isHuman = false;
                $(document).on("mousemove click scroll", function() {
                    isHuman = true;
                    document.cookie = "_badger_human=1;path=/;max-age=86400";
                });
            })(jQuery);
        ', 'after');
    }

    /**
     * Generate trap paths (vary per page to avoid patterns).
     */
    private function get_trap_paths(): array
    {
        $seed = get_the_ID() ?: time();
        $paths = [];
        
        // Generate 3 unique paths per page
        for ($i = 0; $i < 3; $i++) {
            $hash = substr(md5($seed . $i . wp_salt()), 0, 8);
            $paths[] = self::HOLE_PREFIX . $hash . '/';
        }
        
        return $paths;
    }

    /**
     * Extract trap ID from URI.
     */
    private function extract_trap_id(string $uri): string
    {
        if (preg_match('/' . self::HOLE_PREFIX . '([a-z0-9]+)/', $uri, $matches)) {
            return $matches[1];
        }
        return 'unknown';
    }

    /**
     * Get current honeypot depth for session.
     */
    private function get_honeypot_depth(string $session_id): int
    {
        return (int) get_transient('badger_depth_' . $session_id) ?: 0;
    }

    /**
     * Update honeypot depth for session.
     */
    private function update_honeypot_depth(string $session_id, int $depth): void
    {
        set_transient('badger_depth_' . $session_id, $depth, HOUR_IN_SECONDS);
    }

    /**
     * Calculate time from first page load to honeypot hit.
     */
    private function calculate_time_to_trigger(string $session_id): ?float
    {
        $first_seen = get_transient('badger_first_seen_' . $session_id);
        
        if (!$first_seen) {
            return null;
        }
        
        return round((microtime(true) - $first_seen) * 1000, 2); // milliseconds
    }

    /**
     * Verify if JavaScript was executed.
     */
    private function verify_js_execution(): bool
    {
        return isset($_COOKIE['_badger_js']) && $_COOKIE['_badger_js'] === '1';
    }

    /**
     * Generate unique session ID.
     */
    private function generate_session_id(): string
    {
        return substr(md5(uniqid('', true) . wp_rand()), 0, 16);
    }

    /**
     * Check if current visitor is likely a bot (skip honeypot injection).
     */
    private function is_likely_bot(): bool
    {
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // Common bot signatures that don't need honeypot
        $bot_patterns = [
            'googlebot',
            'bingbot',
            'slurp',
            'duckduckbot',
            'baiduspider',
        ];
        
        foreach ($bot_patterns as $pattern) {
            if (str_contains(strtolower($user_agent), $pattern)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Get client IP address.
     */
    private function get_client_ip(): string
    {
        $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (str_contains($ip, ',')) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                return sanitize_text_field($ip);
            }
        }
        return '';
    }
}
