<?php
/**
 * Per-IP request throttling to limit brute force and DoS.
 */

declare(strict_types=1);

if (!defined('ABSPATH')) {
    exit;
}

class Badger_Rate_Limiter
{
    private static ?self $instance = null;

    public static function get_instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action('init', [$this, 'maybe_throttle'], 0);
    }

    /**
     * Throttle requests when IP exceeds configured limit.
     */
    public function maybe_throttle(): void
    {
        if (!get_option('badger_rate_limit_enabled', false)) {
            return;
        }

        if (is_admin() || wp_doing_cron() || (defined('REST_REQUEST') && REST_REQUEST)) {
            return;
        }

        $ip = $this->get_client_ip();
        if (empty($ip)) {
            return;
        }

        $limit = (int) get_option('badger_rate_limit_requests', 60);
        $window = (int) get_option('badger_rate_limit_window', 60);

        if ($limit <= 0 || $window <= 0) {
            return;
        }

        $key = 'badger_rate_' . md5($ip);
        $data = get_transient($key);

        if ($data === false) {
            $data = ['count' => 1, 'start' => time()];
            set_transient($key, $data, $window);
            return;
        }

        $elapsed = time() - $data['start'];
        if ($elapsed >= $window) {
            $data = ['count' => 1, 'start' => time()];
            set_transient($key, $data, $window);
            return;
        }

        $data['count']++;
        set_transient($key, $data, $window - $elapsed);

        if ($data['count'] > $limit) {
            status_header(429);
            wp_die(
                esc_html__('Too many requests. Please slow down.', 'badger'),
                esc_html__('Rate limit exceeded', 'badger'),
                ['response' => 429]
            );
        }
    }

    private function get_client_ip(): string
    {
        $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (str_contains($ip, ',')) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                return sanitize_text_field($ip);
            }
        }
        return '';
    }
}
